 # -*- coding: utf-8 -*-
"""
--------------------------------------
---------Electromigration UI----------
--------------------------------------
----written by Joeri de Bruijckere----
--------------------------------------
--minor edits by Damian Bouwmeester---
--------------------------------------

Created on Thu Jul 6 14:52:42 2017

Last updated on 08 Mar 2023
"""

import sys, time, os.path, datetime
from PyQt5.QtWidgets import (QApplication, QWidget, QPushButton, QSlider, 
        QFrame, QCheckBox, QVBoxLayout, QHBoxLayout, QGridLayout, QLineEdit, 
        QLabel, QFileDialog, QComboBox)
from PyQt5.QtCore import Qt, QTimer
from PyQt5.QtGui import QColor, QPainter, QPalette
from PyQt5.QtChart import (QChart, QChartView, QLineSeries, QValueAxis, QLogValueAxis)
from ADwin import ADwin, ADwinError
import numpy as np

class Window(QWidget):
    
    def __init__(self):
        super().__init__()
        
        self.init_widgets()
        self.init_layouts()
        self.init_fixed_values()
        self.init_values()
        self.init_timer()
        self.init_connections()
        self.set_theme()
        self.setWindowTitle('Electromigrator v26Aug2018')
        
        self.show()
        
    def init_widgets(self):
        
        # File selection
        self.file_btn = QPushButton('Select Process')
        self.file_le = QLineEdit()
        self.save_btn = QPushButton('Save Data')
        self.save_le = QLineEdit()
        self.saveto_btn = QPushButton('Destination')
        
        # Adwin connection
        self.dev_num_lb = QLabel('ADwin device number:')
        self.dev_num_le = QLineEdit()
        self.check_btn = QPushButton('Connect')
        
        self.range_lb = QLabel('S4C range:')
        self.range_cb = QComboBox()
        self.range_cb.addItem('1 n', 'cali_1n.txt')
        self.range_cb.addItem('1 n 100M', 'cal1n100MDamianTest.txt')
        self.range_cb.addItem('1 n 1G', 'cal1n1GDamianTest.txt')
        self.range_cb.addItem('10 n', 'cali_10n.txt')
        self.range_cb.addItem('10 n 100M', 'cal10n100MDamianTest.txt')
        self.range_cb.addItem('100 n', 'cali_100n.txt')
        self.range_cb.addItem('1 u', 'cali_1u.txt')
        self.range_cb.addItem('1 u 100K', 'cal1u100KDamianTest.txt')
        self.range_cb.addItem('1 u 1M', 'cal1u1MDamianTest.txt')
        self.range_cb.addItem('10 u', 'cali_10u.txt')
        self.range_cb.addItem('100 u', 'cali_100u.txt')
        self.range_cb.addItem('100 u 1k', 'cal100u1KDamianTest.txt')
        self.range_cb.addItem('100 u 10k', 'cal100u10KDamianTest.txt')
        self.range_cb.addItem('1 m ', 'cali_1m.txt')
        self.range_cb.addItem('1 m 100', 'cal1m100DamianTest.txt')
        self.range_cb.addItem('1 m 1k', 'cal1m1kDamianTest.txt')
        self.range_cb.addItem('10 m ', 'cali_10m.txt')
        self.range_cb.addItem('10 m 10', 'cal10m10DamianTest.txt')
        self.range_cb.addItem('10 m 100', 'cal10m100DamianTest.txt')
        self.range_cb.addItem('20 m', 'cali_20m.txt')

        # Settings
        self.trgt_lb = QLabel('Target R (kOhm):')
        self.glb_del_lb = QLabel('Process delay (us):')
        self.adc_del_lb = QLabel('DAC-ADC delay (us):')
        self.trgt_le = QLineEdit()
        self.glb_del_le = QLineEdit()
        self.adc_del_le = QLineEdit()
        self.start_v_lb = QLabel('Start voltage (mV):')
        self.start_v_le = QLineEdit()
        self.int_del_lb = QLabel('Interface delay (ms):')
        self.int_del_le = QLineEdit()
        self.avg_lb = QLabel('Average reading (#):')
        self.avg_le = QLineEdit()
        self.step_lb = QLabel('Ramp step size (uV):')
        self.step_le = QLineEdit()
        self.step_dig_lb = QLabel('Ramp step size (#):')
        self.step_dig_le = QLineEdit()
        
        # Electromigration strategy
        self.cutoff_lb = QLabel('Cut-off criterion:')
        self.cutoff_cb = QComboBox()
        self.cutoff_cb.addItem('|R-R0|/R0', '1')
        self.cutoff_cb.addItem('|R-R0|/(R*R0)', '2')
        self.cutoff_cb.addItem('|dR|', '3')
        self.cutoff_cb.addItem('|dR|/R', '4')
        self.mode_lb = QLabel('AC/DC:')
        self.mode_cb = QComboBox()
        self.mode_cb.addItem('DC', '1')
        self.mode_cb.addItem('AC', '2')
        
        self.mode_cb.setCurrentIndex(0)
        self.cutoff_cb.setCurrentIndex(3)
        self.v_back_ch = QCheckBox('Relative decrease (mV):')
        self.v_back_ch.setDisabled(1)
        self.v_back_le = QLineEdit()
        self.v_DCoff_ch = QCheckBox('DC offset (mV):')
        self.v_DCoff_ch.setDisabled(1)
        self.v_DCoff_le = QLineEdit()

        # Values
        self.curr_lb = QLabel('Current (uA):')
        self.curr_le = QLineEdit()
        self.curr_le.setReadOnly(1)
        self.res_lb = QLabel('Resistance (Ohm):')
        self.res_le = QLineEdit()
        self.res_le.setReadOnly(1)
        self.mdiff_lb = QLabel('Cut-off (%):')
        self.mdiff_le = QLineEdit()
        self.mdiff_le.setReadOnly(1)
        self.endv_lb = QLabel('Voltage (V):')
        self.endv_le = QLineEdit()
        self.endv_le.setReadOnly(1)
        
        # Simulation controls
        self.sim_R_lb = QLabel('Start resistance (Ohm)')
        self.sim_A_lb = QLabel('Start area')
        self.sim_t_lb = QLabel('Threshold (mW)')
        self.sim_R_le = QLineEdit()
        self.sim_A_le = QLineEdit()
        self.sim_t_le = QLineEdit()
        self.sim_Rs_lb = QLabel('Series resistance (Ohm)')
        self.sim_A2_lb = QLabel('Area')
        self.sim_Rs_le = QLineEdit()
        self.sim_A2_le = QLineEdit()
        self.sim_A2_le.setReadOnly(1)
        
        # Control buttons
        self.start_btn = QPushButton('Start')
        self.stop_btn = QPushButton('Stop')
        self.clear_btn = QPushButton('Clear')
        self.auto_btn = QPushButton('Automigrate')
        
        # Voltage slider
        self.volt_lb = QLabel('End voltage (V)')
        self.volt_le = QLineEdit()
        self.volt_sld = QSlider(Qt.Horizontal)
        self.volt_sld.setMinimum(1)
        self.volt_sld.setMaximum(2000)
        self.volt_sld.setTickInterval(200)
        self.volt_sld.setTickPosition(QSlider.TicksBelow)
        
        # Differential slider
        self.diff_lb = QLabel('Cut-off parameter (%)')
        self.diff_le = QLineEdit()
        self.diff_sld = QSlider(Qt.Horizontal)
        self.diff_sld.setMinimum(0)
        self.diff_sld.setMaximum(600)
        self.diff_sld.setTickInterval(100)
        self.diff_sld.setTickPosition(QSlider.TicksBelow)
        
        # IV line Chart
        self.ivchart = QChart()
        self.ivchart.legend().hide()
        self.ivseries = QLineSeries(self.ivchart)
        self.ivseries.setColor(Qt.darkBlue)
        self.ivchart.addSeries(self.ivseries)
        self.ivchartXAxis = QValueAxis()
        self.ivchartXAxis.setTitleText('Bias voltage (V)')
        self.ivchart.addAxis(self.ivchartXAxis, Qt.AlignBottom)
        self.ivchartYAxis = QValueAxis()
        self.ivchartYAxis.setTitleText('Current (uA)')
        self.ivchartYAxis.applyNiceNumbers()
        self.ivchart.addAxis(self.ivchartYAxis, Qt.AlignLeft)
        self.ivseries.attachAxis(self.ivchartXAxis)
        self.ivseries.attachAxis(self.ivchartYAxis)
        self.ivchartView = QChartView(self.ivchart)
        self.ivchartView.setMinimumWidth(500)
        self.ivchartView.setRenderHint(QPainter.Antialiasing, 0)
        
        # Conductance line Chart
        self.rchart = QChart()
        self.rchart.legend().hide()
        self.rseries = QLineSeries(self.rchart)
        self.rseries.setColor(Qt.darkBlue)
        self.rchart.addSeries(self.rseries)
        self.rchartXAxis = QValueAxis()
        self.rchartYAxis = QLogValueAxis()
        self.rchartYAxis.setBase(10.0)
        self.rchartYAxis.setMinorTickCount(-1)
        self.rchartXAxis.setTitleText('Time (a.u.)')
        self.rchartYAxis.setTitleText('Resistance (Ohm)')
        self.rchart.addAxis(self.rchartXAxis, Qt.AlignBottom)
        self.rchart.addAxis(self.rchartYAxis, Qt.AlignLeft)
        self.rseries.attachAxis(self.rchartXAxis)
        self.rseries.attachAxis(self.rchartYAxis)
        self.rchartView = QChartView(self.rchart)
        self.rchartView.setMinimumWidth(500)
        self.rchartView.setRenderHint(QPainter.Antialiasing, 0)
        
        # Horizontal lines
        self.hline = self.create_hline()
        self.hline2 = self.create_hline()
        self.hline3 = self.create_hline()
        self.hline4 = self.create_hline()
        
        # Checkboxes
        self.sim_ch = QCheckBox('ADwin simulation')
        self.sim_ch.setChecked(0)
        self.singleramp_ch = QCheckBox('Single ramp Adwin process')
        self.singleramp_ch.setChecked(1)
 
    def create_hline(self):
        hline = QFrame()
        hline.setFrameShape(QFrame.HLine)
        hline.setFrameShadow(QFrame.Sunken)
        return hline
        
    def init_layouts(self):
        
        grid_files = QGridLayout()
        grid_files.addWidget(self.file_btn, 1, 1)
        grid_files.addWidget(self.file_le, 1, 2, 1, 3)
        grid_files.addWidget(self.saveto_btn, 2, 1)
        grid_files.addWidget(self.save_le, 2, 2)
        grid_files.addWidget(self.save_btn, 2, 3)
        
        hbox_conn = QHBoxLayout()
        hbox_conn.addWidget(self.range_lb)
        hbox_conn.addWidget(self.range_cb)
        hbox_conn.addStretch()
        hbox_conn.addWidget(self.dev_num_lb)
        hbox_conn.addWidget(self.dev_num_le)
        hbox_conn.addWidget(self.check_btn)
        
        grid_pars = QGridLayout()
        grid_pars.addWidget(self.glb_del_lb, 1, 1)
        grid_pars.addWidget(self.glb_del_le, 1, 2)
        grid_pars.addWidget(self.adc_del_lb, 1, 3)
        grid_pars.addWidget(self.adc_del_le, 1, 4)
        grid_pars.addWidget(self.start_v_lb, 2, 1)
        grid_pars.addWidget(self.start_v_le, 2, 2)
        grid_pars.addWidget(self.int_del_lb, 2, 3)
        grid_pars.addWidget(self.int_del_le, 2, 4)
        grid_pars.addWidget(self.trgt_lb, 3, 1)
        grid_pars.addWidget(self.trgt_le, 3, 2)
        grid_pars.addWidget(self.avg_lb, 3, 3)
        grid_pars.addWidget(self.avg_le, 3, 4)
        grid_pars.addWidget(self.step_lb, 4, 1)
        grid_pars.addWidget(self.step_le, 4, 2)
        grid_pars.addWidget(self.step_dig_lb, 4, 3)
        grid_pars.addWidget(self.step_dig_le, 4, 4)
        
        grid_ctrl = QGridLayout()
        grid_ctrl.addWidget(self.singleramp_ch, 1, 1)
        grid_ctrl.addWidget(self.v_back_ch, 1, 2)
        grid_ctrl.addWidget(self.v_back_le, 1, 3)
        grid_ctrl.addWidget(self.cutoff_lb, 2, 1)
        grid_ctrl.addWidget(self.cutoff_cb, 2, 2)
        grid_ctrl.addWidget(self.mode_lb, 3, 1)
        grid_ctrl.addWidget(self.mode_cb, 3, 2)
        grid_ctrl.addWidget(self.v_DCoff_ch, 3, 3)
        grid_ctrl.addWidget(self.v_DCoff_le, 3, 4)
        
        
        
        grid_sim = QGridLayout()
        grid_sim.addWidget(self.sim_ch, 1, 1)
        grid_sim.addWidget(self.sim_R_lb, 2, 1)
        grid_sim.addWidget(self.sim_Rs_lb, 3, 1)
        grid_sim.addWidget(self.sim_R_le, 2, 2)
        grid_sim.addWidget(self.sim_Rs_le, 3, 2)
        grid_sim.addWidget(self.sim_t_lb, 1, 3)
        grid_sim.addWidget(self.sim_A_lb, 2, 3)
        grid_sim.addWidget(self.sim_A2_lb, 3, 3)
        grid_sim.addWidget(self.sim_t_le, 1, 4)
        grid_sim.addWidget(self.sim_A_le, 2, 4)
        grid_sim.addWidget(self.sim_A2_le, 3, 4)
        
        grid_mon = QGridLayout()
        grid_mon.addWidget(self.curr_lb, 1, 1)
        grid_mon.addWidget(self.curr_le, 1, 2)
        grid_mon.addWidget(self.res_lb, 1, 3)
        grid_mon.addWidget(self.res_le, 1, 4)
        grid_mon.addWidget(self.endv_lb, 2, 1)
        grid_mon.addWidget(self.endv_le, 2, 2)        
        grid_mon.addWidget(self.mdiff_lb, 2, 3)
        grid_mon.addWidget(self.mdiff_le, 2, 4)
        
        hbox_io = QHBoxLayout()
        hbox_io.addWidget(self.start_btn)
        hbox_io.addWidget(self.stop_btn)
        hbox_io.addWidget(self.clear_btn)
        #hbox_io.addWidget(self.auto_btn)
        hbox_io.addStretch()
        
        hbox_vslider = QHBoxLayout()
        hbox_vslider.addWidget(self.volt_sld, 5)
        hbox_vslider.addWidget(self.volt_le, 1)
        
        hbox_dslider = QHBoxLayout()
        hbox_dslider.addWidget(self.diff_sld, 5)
        hbox_dslider.addWidget(self.diff_le, 1)
        
        control_panel = QVBoxLayout()
        control_panel.addLayout(grid_files)
        control_panel.addWidget(self.hline)
        control_panel.addLayout(hbox_conn)
        control_panel.addLayout(grid_pars)
        control_panel.addWidget(self.hline2)
        control_panel.addLayout(grid_ctrl)
        control_panel.addWidget(self.hline3)
        control_panel.addLayout(grid_sim)
        control_panel.addWidget(self.hline4)
        control_panel.addLayout(grid_mon)
        control_panel.addWidget(self.volt_lb)
        control_panel.addLayout(hbox_vslider)
        control_panel.addWidget(self.diff_lb)
        control_panel.addLayout(hbox_dslider)
        control_panel.addLayout(hbox_io)
        control_panel.addStretch()
        
        overall_layout = QHBoxLayout()
        overall_layout.addLayout(control_panel, 1)
        overall_layout.addWidget(self.ivchartView, 4)
        overall_layout.addWidget(self.rchartView, 4)
        
        self.setLayout(overall_layout)        
    
    def init_timer(self):
        self.ui_timer = QTimer()
        self.ui_timer.setInterval(int(self.int_del_le.text()))
        self.ui_timer.timeout.connect(self.update_process)
        self.ctrl_timer = QTimer()
        self.ctrl_timer.timeout.connect(self.controller_loop)
    
    def init_fixed_values(self): # Do not reset when process is stopped
        self.file_le.setText('EM_v31July2018.t91')
        self.save_le.setText('EM_test')
        self.range_cb.setCurrentIndex(14)
        self.dev_num_le.setText('1')
        self.trgt_le.setText('50.000')
        self.glb_del_le.setText('25')
        self.adc_del_le.setText('7')
        self.start_v_le.setText('200')
        self.int_del_le.setText('5')
        self.step_le.setText('305.175')
        self.step_dig_le.setText('1')
        self.v_back_le.setText('10')
        self.v_DCoff_le.setText('200')
        self.sim_R_le.setText('100.000')
        self.sim_Rs_le.setText('80.000')
        self.sim_A_le.setText('1.000')
        self.sim_t_le.setText('1.000')
        self.avg_le.setText('1')
        self.ilist = []
        self.vlist = []
        self.current_array = np.array([])
        self.boot = 0
        
    def init_values(self): # Reset when process is stopped
        self.curr_le.setDisabled(1)
        self.res_le.setDisabled(1)
        self.mdiff_le.setDisabled(1)
        self.endv_le.setDisabled(1)
        self.volt_sld.setValue(1)
        self.volt_le.setText('0.25')
        self.diff_sld.setValue(0)
        self.diff_le.setText('1.0')
        self.start_btn.setEnabled(1)
        self.stop_btn.setDisabled(1)
        self.min_current = 1e10
        self.min_voltage = 1e10
        self.max_current = 0.0
        self.max_voltage = 0.0
        
        self.max_conductance = 0.0
        self.timecount = 0.0
        self.sign = 0
        
    def init_connections(self):
  
        self.file_btn.clicked.connect(self.file_btn_clk)
        self.save_btn.clicked.connect(self.save_btn_clk)
        self.saveto_btn.clicked.connect(self.saveto_btn_clk)
        self.check_btn.clicked.connect(self.check_btn_clk)
        
        self.range_cb.currentIndexChanged.connect(self.range_select)
        
        self.volt_sld.valueChanged.connect(self.volt_change)
        self.diff_sld.valueChanged.connect(self.diff_change)
        
        self.start_btn.clicked.connect(self.start_btn_clk)
        self.stop_btn.clicked.connect(self.stop_btn_clk)
        self.clear_btn.clicked.connect(self.clear_window)
        self.auto_btn.clicked.connect(self.init_controller)
        
        self.sim_ch.toggled.connect(self.simulation_toggle)
        self.simulation_toggle()
        
        self.v_back_ch.toggled.connect(self.v_back_toggle)
        self.v_back_toggle()
        
        self.volt_le.editingFinished.connect(self.volt_le_edit)
        self.diff_le.editingFinished.connect(self.diff_le_edit)
        
        self.singleramp_ch.toggled.connect(self.singleramp_toggle)
        self.step_le.editingFinished.connect(self.edit_step)
        self.step_dig_le.editingFinished.connect(self.edit_step_dig)
     
    def set_theme(self):
#        for widget in self.window().children():
#            if isinstance(widget, QLineEdit):
#                #widget.setStyleSheet('background-color: black; color: white')
#                pass
        pal = self.window().palette()
        pal.setColor(QPalette.Window, QColor(0x40434a))
        pal.setColor(QPalette.WindowText, QColor(0xd6d6d6))
        self.window().setPalette(pal)
        self.ivchartView.chart().setTheme(QChart.ChartThemeBlueCerulean)
        self.rchartView.chart().setTheme(QChart.ChartThemeBlueCerulean)
        
    def file_btn_clk(self):
        filename = QFileDialog.getOpenFileName(self, 'Open .t91 File')
        print(filename[0])
        self.file_le.setText(filename[0])
    
    def save_btn_clk(self):
        ivdata = np.array([self.vlist, self.ilist])
        ivdata = ivdata.T
        if self.sim_ch.isChecked():
            np.savetxt(self.save_le.text()+'_sim_'+
                       '{:%d%b%Y_%H%M%S}'.format(datetime.datetime.now())+'.txt', ivdata)
        else:
            np.savetxt(self.save_le.text()+'_'+
                       '{:%d%b%Y_%H%M%S}'.format(datetime.datetime.now())+'.txt', ivdata)
    
    def saveto_btn_clk(self):
        filename = QFileDialog.getSaveFileName(self, 'Save data file')
        self.save_le.setText(filename[0])
    
    def range_select(self):
        table_filename = self.range_cb.itemData(self.range_cb.currentIndex())
        if os.path.isfile(table_filename):
            print('Conversion table '+table_filename+' selected')
        else:
            print('Error: Conversion table '+table_filename+' not found')
    
    def volt_change(self):
        volt_value = self.volt_sld.value()*0.001
        volt_str = '%.3f' % volt_value
        self.volt_le.setText(volt_str)     
        
    def diff_change(self):
        diff_value = '%.3f' % (10.0**(self.diff_sld.value()*0.01-3.0))
        self.diff_le.setText(diff_value)
        
    def volt_le_edit(self):
        self.volt_sld.setValue(int(float(self.volt_le.text())*1000))
        
    def diff_le_edit(self):
        self.diff_sld.setValue(int(100*(np.log10(float(self.diff_le.text()))+3)))

    def start_btn_clk(self):
        if self.start_btn.text() == 'Start':
            self.start_process()

    def stop_btn_clk(self):
        if self.start_btn.text() == 'Running...':
            self.stop_process()
            if self.vlist:
                self.save_btn_clk()

    def clear_window(self):
        if self.start_btn.text() == 'Running...':
            self.stop_btn_clk()
        else:
            self.init_values()        
        self.ivseries.clear()
        self.rseries.clear()
        self.ilist = []
        self.vlist = []
        self.sim_A2_le.setText('')
        self.res_le.setText('')
        self.curr_le.setText('')
        self.mdiff_le.setText('')
        self.endv_le.setText('')
            
    def simulation_toggle(self):
        check = self.sim_ch.isChecked()
        self.sim_R_le.setEnabled(check)
        self.sim_Rs_le.setEnabled(check)
        self.sim_A_le.setEnabled(check)
        self.sim_A2_le.setEnabled(check)
        self.sim_t_le.setEnabled(check)
     
    def v_back_toggle(self):
        check1 = self.v_back_ch.isChecked()
        check2 = not self.singleramp_ch.isChecked()
        self.v_back_le.setEnabled(check1*check2)
        
    def singleramp_toggle(self):
        check = self.singleramp_ch.isChecked()
        self.v_back_le.setDisabled(check)
        self.v_back_ch.setDisabled(check)
        self.v_back_toggle()
        
    def edit_step(self):
        step_digits = np.ceil(float(self.step_le.text())/305.175) 
        self.step_le.setText('%.3f' % (step_digits*305.175)) 
        self.step_dig_le.setText('%d' % step_digits)
        
    def edit_step_dig(self):
        step_digits = np.ceil(float(self.step_dig_le.text())) 
        self.step_le.setText('%.3f' % (step_digits*305.175)) 
        self.step_dig_le.setText('%d' % step_digits)
        
    def check_btn_clk(self):
        self.boot = self.boot_ADwin()
        
    def start_process(self):
        self.boot = self.boot_ADwin()
        if self.boot == 1:
            self.start_btn.setText('Running...')
            self.start_btn.setDisabled(1)
            self.stop_btn.setEnabled(1)
            self.curr_le.setEnabled(1)
            self.res_le.setEnabled(1)
            self.load_process()
            self.write_init_pars()
            self.adw.Start_Process(1)
            time.sleep(0.5)
            self.ui_timer.setInterval(int(self.int_del_le.text()))
            self.ui_timer.start()
        
    def stop_process(self):
        self.ui_timer.stop()
        self.ctrl_timer.stop()
        self.adw.Stop_Process(1)
        self.start_btn.setText('Start')
        self.init_values()
        print('Process stopped...')
        if self.vlist:
            self.save_btn_clk()
        
    def boot_ADwin(self):
        device_number = int(self.dev_num_le.text())
        processor_type = '9'
        raise_exceptions = 1
        try:
            if self.sim_ch.isChecked() == 1:
                if self.sim_A2_le.text() == '': # This field is empty if Wire already exists
                    self.adw = ADwin_simulation(device_number, raise_exceptions)    
                    A = float(self.sim_A_le.text())
                    R = float(self.sim_R_le.text())
                    t = 0.001*float(self.sim_t_le.text())
                    Rs = float(self.sim_Rs_le.text())
                    self.adw.wire = Wire(R, A, t, Rs)
                else:
                    pass
            else:
                self.adw = ADwin(device_number, raise_exceptions)
            print('Booting ADwin system...')
            self.adw.Boot(self.adw.ADwindir+'adwin'+processor_type+'.btl')
            self.check_btn.setText('Connected')
            return 1
        except ADwinError as e:
            print('***', e)
            self.check_btn.setText('Connect')
            return 0
        
    def load_process(self):
        self.adw.Load_Process(self.file_le.text())
        print('Process loaded...')
        
    def write_init_pars(self):
        # Convert process parameters to ADwin units
        process_delay = int(40*float(self.glb_del_le.text()))
        self.start_voltage = 0.001*float(self.start_v_le.text())
        start_value = round(self.start_voltage*3276.8+32768)
        end_voltage = float(self.volt_le.text())
        end_value = round(end_voltage*3276.8+32768)
        v_back = int(self.v_back_ch.isChecked()*float(self.v_back_le.text())/0.305)
        v_DCoff = int(self.v_DCoff_ch.isChecked()*float(self.v_DCoff_le.text())/0.305)
        self.target_r = float(self.trgt_le.text())*1000
        
        # Write process parameters to ADwin
        self.adw.Set_Processdelay(1, process_delay)
        self.adw.Set_Par(1, start_value)                            # Start voltage in Adwin units
        self.adw.Set_Par(2, end_value)                              # Cut-off voltage in Adwin units
        self.adw.Set_Par(4, int(float(self.adc_del_le.text())*10))  # DAC-ADC delay in Adwin units: 0.1 us
        self.adw.Set_Par(5, v_back)                                 # Amount by which voltage is swept back after cutoff in Adwin units
        self.adw.Set_Par(6, int(self.avg_le.text()))                # Number of times ReadADC is averaged
        self.adw.Set_Par(7, int(self.cutoff_cb.currentIndex()))     # Cut-off criterion
        self.adw.Set_Par(8, int(self.singleramp_ch.isChecked()))    # Single ramp process (1 = single ramp process (stop process at every cutoff), 0 = continuous (only stop process at target R))
        self.adw.Set_Par(9, int(self.step_dig_le.text()))           # Step size voltage ramp in Adwin units
        self.adw.Set_Par(11, self.sign)                             # AC sign(0 -> 1 , 1 -> -1)
        self.adw.Set_Par(12, v_DCoff)                               # DC offset
        self.adw.Set_FPar(1, self.target_r)                         # Cut-off resistance in Ohm
        self.adw.Set_FPar(2, float(self.diff_le.text())*0.01)       # Maximum change in resistance (>=1)
        table_file = self.range_cb.itemData(self.range_cb.currentIndex())
        if os.path.isfile(table_file):
            print('Conversion table '+table_file+' found...')
            table_data = np.loadtxt(table_file)
            table_list = table_data.tolist()
            self.adw.SetData_Float(table_list, 1, 1, len(table_list))   # Look-up table current conversion
            print('Conversion table '+table_file+' uploaded to Adwin...')
        else:
            print('Error: Conversion table '+table_file+' not found...')
        time.sleep(0.1)
                    
    def update_process(self):
        if self.singleramp_ch.isChecked() == 1: # Single ramp process 
            if self.adw.Get_Par(10) == 0:
                self.update_ui()
                if self.resistance > self.target_r:
                    self.stop_process()
                else:
                    self.set_parameters()
                    self.adw.Start_Process(1)
        else: # Continuous ramp loop
            if self.adw.Get_Par(10) == 1:
                self.set_parameters()
            else:
                self.stop_process()
            self.update_ui()

    def update_ui(self):
        self.get_parameters()
        self.update_chart()
        self.update_rchart()
        self.curr_le.setText('%.3f' % self.current)         # Display current in uA
        self.res_le.setText('%.3f' % self.resistance)       # Display resistance in Ohm
        self.mdiff_le.setText('%.3f' % self.cutoff)         # Display cut-off in %
        self.endv_le.setText('%.3f' % self.voltage)         # Display voltage in V
        if self.sim_ch.isChecked() == 1:
            self.sim_A2_le.setText('%.3f' % self.adw.wire.A)
#        if self.current_array.size < 1000:
#            self.current_array = np.append(self.current_array, float(self.current))
#        else:
#            std_dev = np.std(self.current_array)
#            print(std_dev)
#            self.current_array = np.array([])
        
    def update_chart(self):
        if self.voltage > self.max_voltage:
            self.max_voltage = self.voltage
        if self.voltage < self.min_voltage:
            self.min_voltage = self.voltage

        if self.current > self.max_current:
            self.max_current = self.current
        if self.current < self.min_current:
            self.min_current = self.current

        self.ivchart.axisX().setRange(self.min_voltage, self.max_voltage*1.05)
        self.ivchart.axisY().setRange(self.min_current, self.max_current*1.05)
        self.ivseries.append(self.voltage, self.current)
        self.vlist.append(self.voltage)
        self.ilist.append(self.current)
        if len(self.vlist) > 1000:
            self.ivseries.remove(0)
            
    def update_rchart(self):
        if self.resistance > 20.0 and self.resistance < self.target_r:
            self.rseries.append(self.timecount, self.resistance)
            self.rchart.axisX().setRange(0.0, self.timecount)
            self.rchart.axisY().setRange(20.0, self.target_r)
        self.timecount += 1.0
        if len(self.vlist) > 1000:
            self.rseries.remove(0)
        
    def set_parameters(self):
        
        if int(self.mode_cb.currentIndex())==1:     #AC mode
            self.sign=(self.sign + 1) % 2
        start_value = round( (-1) ** self.sign * self.start_voltage * 3276.8 + 32768 )
        end_value = np.ceil( (-1) ** self.sign * float( self.volt_le.text() ) * 3276.8 + 32768 )
        diff = float(self.diff_le.text())*0.01
        self.adw.Set_Par(1, int(start_value))       # Start voltage
        self.adw.Set_Par(2, int(end_value))         # End voltage
        self.adw.Set_Par(11, self.sign)             # AC sign
        self.adw.Set_FPar(2, diff)                  # Cut-off criterion
        
    def get_parameters(self):
        self.voltage = (self.adw.Get_Par(3)-32768)/3276.8   # Get end voltage in V from Adwin
        self.resistance = self.adw.Get_FPar(3)              # End resistance in Ohm
        self.current = 1e6*self.voltage/self.resistance     # End current in uA
        self.cutoff = self.adw.Get_FPar(4)                  # End cut-off in %

    def init_controller(self):
        self.clear_window()
        self.start_process()
        self.control_status = 0
        self.ui_timer.stop()
        self.ctrl_timer.setInterval(int(self.int_del_le.text()))
        self.ctrl_timer.start()                

    def controller_loop(self):
        if self.adw.Get_Par(10) == 0:
            if self.control_status == 0: # 0 = read, 1 = write
                self.update_ui()
                self.control_status = 1
            else:
                v = self.volt_sld.value()
                if self.voltage < v: 
                    self.volt_sld.setValue(v-1)
                else:
                    self.diff_sld.setValue(self.diff_sld.value()+1)
                self.set_parameters()
                self.adw.Start_Process(1)
                self.control_status = 0
        
      
class ADwin_simulation:
    
    def __init__(self, DeviceNo = 1, raiseExceptions = 1):
        self.ADwindir = 'ADwindir'
        self.process_status = 0
        self.pars = [None] * 12
        self.fpars = [None] * 4
        self.fdata = [None]
        self.adwin_timer = QTimer()
        
    def Boot(self, Filename):
        print('ADwin: '+Filename+' booted...')
        
    def Load_Process(self, Filename):
        self.adwin_timer.timeout.connect(self.main_loop)
        print('ADwin: Process loaded: '+Filename)
        
    def Start_Process(self, ProcessNo):
        self.process_status = 1
        self.pars[9] = 1
        self.init_process()
        self.adwin_timer.start()
        #print('ADwin: Process '+str(ProcessNo)+' started...')

    def Stop_Process(self, ProcessNo):
        self.process_status = 0
        self.adwin_timer.stop()
        #print('ADwin: Process '+str(ProcessNo)+' stopped...')

    def Clear_Process(self, ProcessNo):
        self.process_status = 0
        self.adwin_timer.stop()
        print('ADwin: Process '+str(ProcessNo)+' cleared...')

    def Process_Status(self, ProcessNo):
        return self.process_status

    def Set_Processdelay(self, ProcessNo, Processdelay):
        interval = round(Processdelay/40000)
        self.adwin_timer.setInterval(interval)
        print('ADwin: Global delay '+'set to '+str(Processdelay))
        
    def Set_Par(self, Index, Value):
        Index = Index-1
        self.pars[Index] = Value
        
    def Get_Par(self, Index):
        Index = Index-1
        return self.pars[Index]

    def Get_Par_Block(self, StartIndex, Count):
        StartIndex = StartIndex - 1
        block = [self.pars[x] for x in range(StartIndex, StartIndex+Count)]
        return block

    def Get_Par_All(self):
        return self.pars

    def Set_FPar(self, Index, Value):
        Index = Index-1
        self.fpars[Index] = Value

    def Get_FPar(self, Index):
        Index = Index-1
        return self.fpars[Index]
        
    def Get_FPar_Block(self, StartIndex, Count):
        StartIndex = StartIndex - 1
        block = [self.fpars[x] for x in range(StartIndex, StartIndex+Count)]
        return block

    def Get_FPar_All(self):
        return self.fpars

    def SetData_Long(self, Data, DataNo, StartIndex, Count):
        StartIndex = StartIndex - 1
        DataNo = DataNo - 1
        self.data[DataNo] = [Data[x] for x in range(StartIndex, StartIndex+Count)]

    def GetData_Long(self, DataNo, StartIndex, Count):
        StartIndex = StartIndex - 1
        DataNo = DataNo - 1
        data = [self.data[DataNo][x] for x in range(StartIndex, StartIndex+Count)]
        return data

    def SetData_Float(self, Data, DataNo, StartIndex, Count):
        StartIndex = StartIndex - 1
        DataNo = DataNo - 1
        self.fdata[DataNo] = [Data[x] for x in range(StartIndex, StartIndex+Count)]

    def GetData_Float(self, DataNo, StartIndex, Count):
        StartIndex = StartIndex - 1
        DataNo = DataNo - 1
        data = [self.fdata[DataNo][x] for x in range(StartIndex, StartIndex+Count)]
        return data
        
    def init_process(self):
        self.count = 0
        self.first_loop = 1
        self.current = 0
    
    def main_loop(self):
        v = int(self.pars[0]+self.count)    # New voltage in Adwin units
        self.wire.set_voltage(v)            # Apply voltage to wire
        self.wire.update_resistance()       # Set new resistance wire
        for _ in range(self.pars[5]):
            self.current = self.wire.read_current() + self.current  # Current through wire in A
        self.current = self.current/self.pars[5]
        r = (v-32768)/(self.current*3276.8) # Resistance in Ohm
        self.current = 0 # Clear for next ramp
        
        if self.first_loop == 1:
            self.first_r = r    # First resistance in Ohm
            self.old_r = r
            if r > self.fpars[0]:
                self.finish_process(v, r, 0.0)
                print('Target resistance reached')
            else:
                self.first_loop = 0
        else:
            if self.pars[6] == 0:
                cutoff = abs(r/self.first_r-1)
            elif self.pars[6] == 1:
                cutoff = abs(r-self.first_r)/(r*self.first_r)
            elif self.pars[6] == 2:
                cutoff = abs(r-self.old_r)
                self.old_r = r
            elif self.pars[6] == 3:
                cutoff = abs(r-self.old_r)/r
                self.old_r = r
            
            if ( ((v > self.pars[1]) ^ self.pars[10]) or cutoff > self.fpars[1]): # Check cut-off criteria
                if self.pars[7] == 1:
                    self.finish_process(v, r, cutoff)
                else:
                    if self.pars[4] > 0:         # Step back voltage (if 0, step back to start voltage)
                        self.count = self.count - (-1)**self.pars[10]*self.pars[4]
                        if (-1)**self.pars[10]*self.count < 0:
                            self.count = 0
                    else:
                        self.count = 0
                    self.wire.set_voltage(v)    # Apply voltage to wire
                    self.pars[2] = v            # End voltage in Adwin units
                    self.fpars[2] = r           # End resistance in Adwin units
                    self.fpars[3] = cutoff      # End cut-off in Adwin units 
                    self.first_loop = 1
            else:
                self.count = self.count + (-1)**self.pars[10]*self.pars[8] # Increase voltage ramp counter
                
    def finish_process(self, end_voltage, end_resistance, end_cutoff):
        self.pars[2] = end_voltage       # End voltage in Adwin units
        self.fpars[2] = end_resistance   # End resistance in Adwin units
        self.fpars[3] = end_cutoff       # End cut-off in Adwin units
        self.pars[9] = 0
        self.Stop_Process(1)
               
class Wire:
    
    def __init__(self, resistance, area, threshold, series_resistance):        
        self.R = resistance
        self.A = area
        self.t = threshold
        self.R_series = series_resistance
        
    def set_voltage(self, volt_value):      # Set voltage over wire
        self.V = (volt_value-32768)/3276.8

    def read_current(self):
        current = self.V/self.R
        return current

    def update_resistance(self):
        current = self.V/self.R
        power = current*self.V-self.R_series*current**2
        if power > self.t:
            old_A = self.A
            dA = 0.0005*old_A
            self.A = old_A - dA
            self.R = self.R*(old_A/self.A)
            
            
app = QApplication(sys.argv)    
app.aboutToQuit.connect(app.deleteLater)
w = Window()
app.exec_()
#sys.exit(app.exec_())